'  PCAN-ISO-TP_2004.vb
'
'  ~~~~~~~~~~~~
'
'  PCAN-ISO-TP API
'
'  ~~~~~~~~~~~~
'
'  ------------------------------------------------------------------
'  Author : Fabrice Vergnaud
'  Last changed by:     $Author: Fabrice $
'  Last changed date:   $Date: 2025-01-23 11:32:04 +0100 (Thu, 23 Jan 2025) $
'
'  Language: VB.Net
'  ------------------------------------------------------------------
'
'  Copyright (C) 2015  PEAK-System Technik GmbH, Darmstadt
'  more Info at http:''www.peak-system.com
'
Imports System
Imports System.Text
Imports System.Runtime.InteropServices

''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
'' Inclusion of other needed stuff
''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
Imports TPCANTPHandle = System.UInt16
Imports TPCANTPBitrateFD = System.String

Namespace Peak.Can.IsoTp

#Region "Enumerations"
    ''' <summary>
    ''' Represents a PCAN Baud rate register value
    ''' </summary>
    Public Enum TPCANTPBaudrate As UShort
        ''' <summary>
        ''' 1 MBit/s
        ''' </summary>
        PCANTP_BAUD_1M = &H14
        ''' <summary>
        ''' 800 kBit/s
        ''' </summary>
        PCANTP_BAUD_800K = &H16
        ''' <summary>
        ''' 500 kBit/s
        ''' </summary>
        PCANTP_BAUD_500K = &H1C
        ''' <summary>
        ''' 250 kBit/s
        ''' </summary>
        PCANTP_BAUD_250K = &H11C
        ''' <summary>
        ''' 125 kBit/s
        ''' </summary>
        PCANTP_BAUD_125K = &H31C
        ''' <summary>
        ''' 100 kBit/s
        ''' </summary>
        PCANTP_BAUD_100K = &H432F
        ''' <summary>
        ''' 95,238 kBit/s
        ''' </summary>
        PCANTP_BAUD_95K = &HC34E
        ''' <summary>
        ''' 83,333 kBit/s
        ''' </summary>
        PCANTP_BAUD_83K = &H852B
        ''' <summary>
        ''' 50 kBit/s
        ''' </summary>
        PCANTP_BAUD_50K = &H472F
        ''' <summary>
        ''' 47,619 kBit/s
        ''' </summary>
        PCANTP_BAUD_47K = &H1414
        ''' <summary>
        ''' 33,333 kBit/s
        ''' </summary>
        PCANTP_BAUD_33K = &H8B2F
        ''' <summary>
        ''' 20 kBit/s
        ''' </summary>
        PCANTP_BAUD_20K = &H532F
        ''' <summary>
        ''' 10 kBit/s
        ''' </summary>
        PCANTP_BAUD_10K = &H672F
        ''' <summary>
        ''' 5 kBit/s
        ''' </summary>
        PCANTP_BAUD_5K = &H7F7F
    End Enum

    ''' <summary>
    ''' Represents the different Not Plug-And-Play PCAN Hardware types
    ''' </summary>
    Public Enum TPCANTPHWType As Byte
        ''' <summary>
        ''' PCAN-ISA 82C200
        ''' </summary>
        PCANTP_TYPE_ISA = &H1
        ''' <summary>
        ''' PCAN-ISA SJA1000
        ''' </summary>
        PCANTP_TYPE_ISA_SJA = &H9
        ''' <summary>
        ''' PHYTEC ISA
        ''' </summary>
        PCANTP_TYPE_ISA_PHYTEC = &H4
        ''' <summary>
        ''' PCAN-Dongle 82C200
        ''' </summary>
        PCANTP_TYPE_DNG = &H2
        ''' <summary>
        ''' PCAN-Dongle EPP 82C200
        ''' </summary>
        PCANTP_TYPE_DNG_EPP = &H3
        ''' <summary>
        ''' PCAN-Dongle SJA1000
        ''' </summary>
        PCANTP_TYPE_DNG_SJA = &H5
        ''' <summary>
        ''' PCAN-Dongle EPP SJA1000
        ''' </summary>
        PCANTP_TYPE_DNG_SJA_EPP = &H6
    End Enum

    ''' <summary>
    ''' Represent the PCANTP error and status codes
    ''' </summary>
    Public Enum TPCANTPStatus As Integer
        ''' <summary>
        ''' No error
        ''' </summary>
        PCANTP_ERROR_OK = &H0
        ''' <summary>
        ''' Not Initialized
        ''' </summary>
        PCANTP_ERROR_NOT_INITIALIZED = &H1
        ''' <summary>
        ''' Already Initialized
        ''' </summary>
        PCANTP_ERROR_ALREADY_INITIALIZED = &H2
        ''' <summary>
        ''' Could not obtain memory
        ''' </summary>
        PCANTP_ERROR_NO_MEMORY = &H3
        ''' <summary>
        ''' Input buffer overflow
        ''' </summary>
        PCANTP_ERROR_OVERFLOW = &H4
        ''' <summary>
        ''' No Message available
        ''' </summary>
        PCANTP_ERROR_NO_MESSAGE = &H7
        ''' <summary>
        ''' Wrong message parameters
        ''' </summary>
        PCANTP_ERROR_WRONG_PARAM = &H8
        ''' <summary>
        ''' PCANTP Channel is in BUS-LIGHT error state
        ''' </summary>
        PCANTP_ERROR_BUSLIGHT = &H9
        ''' <summary>
        ''' PCANTP Channel is in BUS-HEAVY error state
        ''' </summary>
        PCANTP_ERROR_BUSHEAVY = &HA
        ''' <summary>
        ''' PCANTP Channel is in BUS-OFF error state
        ''' </summary>
        PCANTP_ERROR_BUSOFF = &HB
        ''' <summary>
        ''' Global CAN error, status code for composition of PCANBasic Errors.
        ''' Remove this value to get a PCAN-Basic TPCANStatus error code.
        ''' </summary>
        PCANTP_ERROR_CAN_ERROR = &H80000000
    End Enum

    ''' <summary>
    ''' Represent message request confirmation values defined in ISO 15765-2
    ''' </summary>
    Public Enum TPCANTPConfirmation As Byte
        ''' <summary>
        ''' No network error
        ''' </summary>
        PCANTP_N_OK = &H0
        ''' <summary>
        ''' timeout occured between 2 frames transmission (sender and receiver side)
        ''' </summary>
        PCANTP_N_TIMEOUT_A = &H1
        ''' <summary>
        ''' sender side timeout while waiting for flow control frame
        ''' </summary>
        PCANTP_N_TIMEOUT_BS = &H2
        ''' <summary>
        ''' receiver side timeout while waiting for consecutive frame
        ''' </summary>
        PCANTP_N_TIMEOUT_CR = &H3
        ''' <summary>
        ''' unexpected sequence number
        ''' </summary>
        PCANTP_N_WRONG_SN = &H4
        ''' <summary>
        ''' invalid or unknown FlowStatus
        ''' </summary>
        PCANTP_N_INVALID_FS = &H5
        ''' <summary>
        ''' unexpected protocol data unit
        ''' </summary>
        PCANTP_N_UNEXP_PDU = &H6
        ''' <summary>
        ''' reception of flow control WAIT frame that exceeds the maximum counter defined by PCANTP_PARAM_WFT_MAX
        ''' </summary>
        PCANTP_N_WFT_OVRN = &H7
        ''' <summary>
        ''' buffer on the receiver side cannot store the data length (server side only)
        ''' </summary>
        PCANTP_N_BUFFER_OVFLW = &H8
        ''' <summary>
        ''' general error
        ''' </summary>
        PCANTP_N_ERROR = &H9
        ''' <summary>
        ''' message was invalid and ignored
        ''' </summary>
        PCANTP_N_IGNORED = &HA
    End Enum

    ''' <summary>
    ''' PCANTP parameters
    ''' </summary>
    Public Enum TPCANTPParameter As Byte
        ''' <summary>
        ''' 1 BYTE data describing the block size parameter (BS)
        ''' </summary>
        PCANTP_PARAM_BLOCK_SIZE = &HE1
        ''' <summary>
        ''' 1 BYTE data describing the separation time parameter (STmin)
        ''' </summary>
        PCANTP_PARAM_SEPARATION_TIME = &HE2
        ''' <summary>
        ''' 1 BYTE data describing the debug mode
        ''' </summary>
        PCANTP_PARAM_DEBUG = &HE3
        ''' <summary>
        ''' 1 Byte data describing the condition of a channel
        ''' </summary>
        PCANTP_PARAM_CHANNEL_CONDITION = &HE4
        ''' <summary>
        ''' Integer data describing the Wait Frame Transmissions parameter.
        ''' </summary>
        PCANTP_PARAM_WFT_MAX = &HE5
        ''' <summary>
        ''' 1 BYTE data stating if pending messages are displayed/hidden
        ''' </summary>
        PCANTP_PARAM_MSG_PENDING = &HE6
        ''' <summary>
        ''' PCAN-ISO-TP API version parameter
        ''' </summary>
        PCANTP_PARAM_API_VERSION = &HE7
        ''' <summary>
        ''' 1 BYTE data stating if CAN frame DLC uses padding or not
        ''' </summary>
        PCANTP_PARAM_CAN_DATA_PADDING = &HE8
        ''' <summary>
        '''  1 BYTE data stating if unsegmented (NON-ISO-TP) CAN frames can be received
        ''' </summary>
        PCANTP_PARAM_CAN_UNSEGMENTED = &HE9
        ''' <summary>
        ''' PCAN-ISO-TP receive event handler parameter
        ''' </summary>
        PCANTP_PARAM_RECEIVE_EVENT = &HEA
        ''' <summary>
        ''' 1 BYTE data stating the value used for CAN data padding
        ''' </summary>
        PCANTP_PARAM_PADDING_VALUE = &HED
        ''' <summary>
        ''' 1 BYTE data stating the default priority value for normal fixed, mixed And enhanced addressing (default=6)
        ''' </summary>
        PCANTP_PARAM_J1939_PRIORITY = &HEE
        ''' <summary>
        ''' 1 BYTE data stating the default DLC to use when transmitting messages with CAN FD
        ''' </summary>
        PCANTP_PARAM_CAN_TX_DL = &HEF
        ''' <summary>
        ''' 1 BYTE data stating how to optimize Minimum SeparationTime latency (enabled=1 by default).
        ''' Set value to 0 to ensure STMin is always strictly respected (but consequently lose some speed in communication process).
        ''' </summary>
        PCANTP_PARAM_SEPARATION_TIME_OPTIMIZATION = &HF0
    End Enum

    ''' <summary>
    ''' PCANTP message types
    ''' </summary>
    Public Enum TPCANTPMessageType As Byte
        ''' <summary>
        ''' Unknown (non-ISO-TP) message
        ''' </summary>
        PCANTP_MESSAGE_UNKNOWN = &H0
        ''' <summary>
        ''' Diagnostic Request/Confirmation
        ''' </summary>
        PCANTP_MESSAGE_DIAGNOSTIC = &H1
        ''' <summary>
        ''' Remote Dignostic Request/Confirmation (uses RA address)
        ''' </summary>
        PCANTP_MESSAGE_REMOTE_DIAGNOSTIC = &H2
        ''' <summary>
        ''' Confirms that a message has been sent successfully/ not successfully
        ''' </summary>
        PCANTP_MESSAGE_REQUEST_CONFIRMATION = &H3
        ''' <summary>
        ''' Multi-Frame Message is being received
        ''' </summary>
        PCANTP_MESSAGE_INDICATION = &H4
        ''' <summary>
        ''' Multi-Frame Message is being transmitted
        ''' </summary>
        PCANTP_MESSAGE_INDICATION_TX = &H5
    End Enum

    ''' <summary>
    ''' PCANTP message types
    ''' </summary>
    <Flags()> _
    Public Enum TPCANTPIdType As Byte
        ''' <summary>
        ''' 11 bits CAN ID (CAN Standard Frame)
        ''' </summary>
        PCANTP_ID_CAN_11BIT = &H1
        ''' <summary>
        ''' 29 bits CAN ID (CAN Extended Frame)
        ''' </summary>
        PCANTP_ID_CAN_29BIT = &H2
        ''' <summary>
        ''' CAN FD flag
        ''' </summary>
        PCANTP_ID_CAN_FD = &H4
        ''' <summary>
        ''' Bitrate Switch flag (only if CAN FD)
        ''' </summary>
        PCANTP_ID_CAN_BRS = &H8
    End Enum
    ''' <summary>
    ''' PCANTP message types utility function to handle priority J1939 compatible ISO-TP messages
    ''' </summary>
    Public Class TPCANTPIdTypePriority
        ''' <summary>
        ''' Get the 29bits CAN ID type with a specific J1939 priority
        ''' </summary>
        Public Shared Function PCANTP_ID_CAN_GET_29B(ByVal j1939_priority As Byte) As TPCANTPIdType
            Return (CByte(j1939_priority << 5) Or CanTpApi.PCANTP_ID_CAN_IS_PRIORITY_MASK Or (CByte(TPCANTPIdType.PCANTP_ID_CAN_29BIT) And CanTpApi.PCANTP_ID_CAN_MASK))
        End Function
        ''' <summary>
        ''' Retrieves the priority field from a CAN ID type
        ''' </summary>
        Public Shared Function PCANTP_ID_CAN_GET_PRIORIY(ByVal id_type As Byte) As Byte
            Return CByte(id_type >> 5)
        End Function
        ''' <summary>
        ''' States if the CAN ID Type is 29bits
        ''' </summary>
        Public Shared Function PCANTP_ID_CAN_IS_EXTENDED(ByVal id_type As Byte) As Boolean
            Return ((id_type And CByte(TPCANTPIdType.PCANTP_ID_CAN_29BIT)) = CByte(TPCANTPIdType.PCANTP_ID_CAN_29BIT))
        End Function
        ''' <summary>
        ''' States if the id_type contains a J1939 priority field
        ''' </summary>
        Public Shared Function PCANTP_ID_CAN_HAS_PRIORITY(ByVal id_type As Byte) As Boolean
            Return (PCANTP_ID_CAN_IS_EXTENDED(id_type) AndAlso ((id_type And CanTpApi.PCANTP_ID_CAN_IS_PRIORITY_MASK) = CanTpApi.PCANTP_ID_CAN_IS_PRIORITY_MASK))
        End Function
    End Class

    ''' <summary>
    ''' PCANTP addressing type
    ''' </summary>
    Public Enum TPCANTPFormatType As Byte
        ''' <summary>
        ''' unknown adressing format
        ''' </summary>
        PCANTP_FORMAT_UNKNOWN = &HFF
        ''' <summary>
        ''' unsegmented CAN frame
        ''' </summary>
        PCANTP_FORMAT_NONE = &H0
        ''' <summary>
        ''' normal adressing format from ISO 15765-2
        ''' </summary>
        PCANTP_FORMAT_NORMAL = &H1
        ''' <summary>
        ''' fixed normal adressing format from ISO 15765-2
        ''' </summary>
        PCANTP_FORMAT_FIXED_NORMAL = &H2
        ''' <summary>
        ''' extended adressing format from ISO 15765-2
        ''' </summary>
        PCANTP_FORMAT_EXTENDED = &H3
        ''' <summary>
        ''' mixed adressing format from ISO 15765-2
        ''' </summary>
        PCANTP_FORMAT_MIXED = &H4
        ''' <summary>
        ''' enhanced adressing format from ISO 15765-3
        ''' </summary>
        PCANTP_FORMAT_ENHANCED = &H5
    End Enum

    ''' <summary>
    ''' PCANTP addressing type
    ''' </summary>
    Public Enum TPCANTPAddressingType As Byte
        ''' <summary>
        ''' Unknown (non-ISO-TP) message
        ''' </summary>
        PCANTP_ADDRESSING_UNKNOWN = &H0
        ''' <summary>
        ''' Physical addressing
        ''' </summary>
        PCANTP_ADDRESSING_PHYSICAL = &H1
        ''' <summary>
        ''' Functional addressing
        ''' </summary>
        PCANTP_ADDRESSING_FUNCTIONAL = &H2
    End Enum

#End Region

#Region "Structures"
    Public Structure TPCANTPMsg
        ''' <summary>
        ''' Represents the origin of this message (address from
        ''' where this message was or will be sent)
        ''' </summary>
        Public SA As Byte
        ''' <summary>
        ''' Represents the destination of this message (address to
        ''' where this message was or will be sent)
        ''' </summary>
        Public TA As Byte
        ''' <summary>
        ''' Represents the kind of addressing being used for communication
        ''' </summary>
        <MarshalAs(UnmanagedType.U1)> _
        Public TA_TYPE As TPCANTPAddressingType
        ''' <summary>
        ''' Represents the destination of this message in a remote network
        ''' </summary>
        Public RA As Byte

        ''' <summary>
        ''' Identifies the kind of CAN Identifier
        ''' </summary>
        <MarshalAs(UnmanagedType.U1)> _
        Public IDTYPE As TPCANTPIdType
        ''' <summary>
        ''' Identifies the kind of data represented by this message
        ''' </summary>
        <MarshalAs(UnmanagedType.U1)> _
        Public MSGTYPE As TPCANTPMessageType
        ''' <summary>
        ''' Represents the kind of format represented by this message
        ''' </summary>
        <MarshalAs(UnmanagedType.U1)> _
        Public FORMAT As TPCANTPFormatType

        ''' <summary>
        ''' Represents the buffer containing the data of this message
        ''' </summary>
        <MarshalAs(UnmanagedType.ByValArray, SizeConst:=4095)> _
        Public DATA As Byte()
        ''' <summary>
        ''' Indicates the length of the useful data within the DATA buffer
        ''' </summary>
        Public LEN As UShort
        ''' <summary>
        ''' Represent message request confirmation
        ''' </summary>
        <MarshalAs(UnmanagedType.U1)> _
        Public RESULT As TPCANTPConfirmation
    End Structure

    ''' <summary>
    ''' Represents a timestamp of a received PCANTP message.
    ''' Total Microseconds = micros + 1000 * millis + 0x100000000 * 1000 * millis_overflow
    ''' </summary>
    Public Structure TPCANTPTimestamp
        ''' <summary>
        ''' Base-value: milliseconds: 0.. 2^32-1
        ''' </summary>
        Public millis As UInt32
        ''' <summary>
        ''' Roll-arounds of millis
        ''' </summary>
        Public millis_overflow As UInt16
        ''' <summary>
        ''' Microseconds: 0..999
        ''' </summary>
        Public micros As UInt16
    End Structure

#End Region

#Region "PCANTP Api"
    ''' <summary>
    ''' The PCAN-ISO-TP API class
    ''' </summary>
    Partial Public NotInheritable Class CanTpApi
#Region "PCAN-BUS Handles Definition"
        ''' <summary>
        ''' Undefined/default value for a PCAN bus
        ''' </summary>
        Public Const PCANTP_NONEBUS As TPCANTPHandle = &H0

        ''' <summary>
        ''' PCAN-ISA interface, channel 1
        ''' </summary>
        Public Const PCANTP_ISABUS1 As TPCANTPHandle = &H21
        ''' <summary>
        ''' PCAN-ISA interface, channel 2
        ''' </summary>
        Public Const PCANTP_ISABUS2 As TPCANTPHandle = &H22
        ''' <summary>
        ''' PCAN-ISA interface, channel 3
        ''' </summary>
        Public Const PCANTP_ISABUS3 As TPCANTPHandle = &H23
        ''' <summary>
        ''' PCAN-ISA interface, channel 4
        ''' </summary>
        Public Const PCANTP_ISABUS4 As TPCANTPHandle = &H24
        ''' <summary>
        ''' PCAN-ISA interface, channel 5
        ''' </summary>
        Public Const PCANTP_ISABUS5 As TPCANTPHandle = &H25
        ''' <summary>
        ''' PCAN-ISA interface, channel 6
        ''' </summary>
        Public Const PCANTP_ISABUS6 As TPCANTPHandle = &H26
        ''' <summary>
        ''' PCAN-ISA interface, channel 7
        ''' </summary>
        Public Const PCANTP_ISABUS7 As TPCANTPHandle = &H27
        ''' <summary>
        ''' PCAN-ISA interface, channel 8
        ''' </summary>
        Public Const PCANTP_ISABUS8 As TPCANTPHandle = &H28

        ''' <summary>
        ''' PPCAN-Dongle/LPT interface, channel 1
        ''' </summary>
        Public Const PCANTP_DNGBUS1 As TPCANTPHandle = &H31

        ''' <summary>
        ''' PCAN-PCI interface, channel 1
        ''' </summary>
        Public Const PCANTP_PCIBUS1 As TPCANTPHandle = &H41
        ''' <summary>
        ''' PCAN-PCI interface, channel 2
        ''' </summary>
        Public Const PCANTP_PCIBUS2 As TPCANTPHandle = &H42
        ''' <summary>
        ''' PCAN-PCI interface, channel 3
        ''' </summary>
        Public Const PCANTP_PCIBUS3 As TPCANTPHandle = &H43
        ''' <summary>
        ''' PCAN-PCI interface, channel 4
        ''' </summary>
        Public Const PCANTP_PCIBUS4 As TPCANTPHandle = &H44
        ''' <summary>
        ''' PCAN-PCI interface, channel 5
        ''' </summary>
        Public Const PCANTP_PCIBUS5 As TPCANTPHandle = &H45
        ''' <summary>
        ''' PCAN-PCI interface, channel 6
        ''' </summary>
        Public Const PCANTP_PCIBUS6 As TPCANTPHandle = &H46
        ''' <summary>
        ''' PCAN-PCI interface, channel 7
        ''' </summary>
        Public Const PCANTP_PCIBUS7 As TPCANTPHandle = &H47
        ''' <summary>
        ''' PCAN-PCI interface, channel 8
        ''' </summary>
        Public Const PCANTP_PCIBUS8 As TPCANTPHandle = &H48
        ''' <summary>
        ''' PCAN-PCI interface, channel 9
        ''' </summary>
        Public Const PCANTP_PCIBUS9 As TPCANTPHandle = &H409
        ''' <summary>
        ''' PCAN-PCI interface, channel 10
        ''' </summary>
        Public Const PCANTP_PCIBUS10 As TPCANTPHandle = &H40A
        ''' <summary>
        ''' PCAN-PCI interface, channel 11
        ''' </summary>
        Public Const PCANTP_PCIBUS11 As TPCANTPHandle = &H40B
        ''' <summary>
        ''' PCAN-PCI interface, channel 12
        ''' </summary>
        Public Const PCANTP_PCIBUS12 As TPCANTPHandle = &H40C
        ''' <summary>
        ''' PCAN-PCI interface, channel 13
        ''' </summary>
        Public Const PCANTP_PCIBUS13 As TPCANTPHandle = &H40D
        ''' <summary>
        ''' PCAN-PCI interface, channel 14
        ''' </summary>
        Public Const PCANTP_PCIBUS14 As TPCANTPHandle = &H40E
        ''' <summary>
        ''' PCAN-PCI interface, channel 15
        ''' </summary>
        Public Const PCANTP_PCIBUS15 As TPCANTPHandle = &H40F
        ''' <summary>
        ''' PCAN-PCI interface, channel 16
        ''' </summary>
        Public Const PCANTP_PCIBUS16 As TPCANTPHandle = &H410

        ''' <summary>
        ''' PCAN-USB interface, channel 1
        ''' </summary>
        Public Const PCANTP_USBBUS1 As TPCANTPHandle = &H51
        ''' <summary>
        ''' PCAN-USB interface, channel 2
        ''' </summary>
        Public Const PCANTP_USBBUS2 As TPCANTPHandle = &H52
        ''' <summary>
        ''' PCAN-USB interface, channel 3
        ''' </summary>
        Public Const PCANTP_USBBUS3 As TPCANTPHandle = &H53
        ''' <summary>
        ''' PCAN-USB interface, channel 4
        ''' </summary>
        Public Const PCANTP_USBBUS4 As TPCANTPHandle = &H54
        ''' <summary>
        ''' PCAN-USB interface, channel 5
        ''' </summary>
        Public Const PCANTP_USBBUS5 As TPCANTPHandle = &H55
        ''' <summary>
        ''' PCAN-USB interface, channel 6
        ''' </summary>
        Public Const PCANTP_USBBUS6 As TPCANTPHandle = &H56
        ''' <summary>
        ''' PCAN-USB interface, channel 7
        ''' </summary>
        Public Const PCANTP_USBBUS7 As TPCANTPHandle = &H57
        ''' <summary>
        ''' PCAN-USB interface, channel 8
        ''' </summary>
        Public Const PCANTP_USBBUS8 As TPCANTPHandle = &H58
        ''' <summary>
        ''' PCAN-USB interface, channel 9
        ''' </summary>
        Public Const PCANTP_USBBUS9 As TPCANTPHandle = &H509
        ''' <summary>
        ''' PCAN-USB interface, channel 10
        ''' </summary>
        Public Const PCANTP_USBBUS10 As TPCANTPHandle = &H50A
        ''' <summary>
        ''' PCAN-USB interface, channel 11
        ''' </summary>
        Public Const PCANTP_USBBUS11 As TPCANTPHandle = &H50B
        ''' <summary>
        ''' PCAN-USB interface, channel 12
        ''' </summary>
        Public Const PCANTP_USBBUS12 As TPCANTPHandle = &H50C
        ''' <summary>
        ''' PCAN-USB interface, channel 13
        ''' </summary>
        Public Const PCANTP_USBBUS13 As TPCANTPHandle = &H50D
        ''' <summary>
        ''' PCAN-USB interface, channel 14
        ''' </summary>
        Public Const PCANTP_USBBUS14 As TPCANTPHandle = &H50E
        ''' <summary>
        ''' PCAN-USB interface, channel 15
        ''' </summary>
        Public Const PCANTP_USBBUS15 As TPCANTPHandle = &H50F
        ''' <summary>
        ''' PCAN-USB interface, channel 16
        ''' </summary>
        Public Const PCANTP_USBBUS16 As TPCANTPHandle = &H510

        ''' <summary>
        ''' PCAN-PC Card interface, channel 1
        ''' </summary>
        Public Const PCANTP_PCCBUS1 As TPCANTPHandle = &H61
        ''' <summary>
        ''' PCAN-PC Card interface, channel 2
        ''' </summary>
        Public Const PCANTP_PCCBUS2 As TPCANTPHandle = &H62

        ''' <summary>
        ''' PCAN-LAN interface, channel 1
        ''' </summary>
        Public Const PCANTP_LANBUS1 As TPCANTPHandle = &H801
        ''' <summary>
        ''' PCAN-LAN interface, channel 2
        ''' </summary>
        Public Const PCANTP_LANBUS2 As TPCANTPHandle = &H802
        ''' <summary>
        ''' PCAN-LAN interface, channel 3
        ''' </summary>
        Public Const PCANTP_LANBUS3 As TPCANTPHandle = &H803
        ''' <summary>
        ''' PCAN-LAN interface, channel 4
        ''' </summary>
        Public Const PCANTP_LANBUS4 As TPCANTPHandle = &H804
        ''' <summary>
        ''' PCAN-LAN interface, channel 5
        ''' </summary>
        Public Const PCANTP_LANBUS5 As TPCANTPHandle = &H805
        ''' <summary>
        ''' PCAN-LAN interface, channel 6
        ''' </summary>
        Public Const PCANTP_LANBUS6 As TPCANTPHandle = &H806
        ''' <summary>
        ''' PCAN-LAN interface, channel 7
        ''' </summary>
        Public Const PCANTP_LANBUS7 As TPCANTPHandle = &H807
        ''' <summary>
        ''' PCAN-LAN interface, channel 8
        ''' </summary>
        Public Const PCANTP_LANBUS8 As TPCANTPHandle = &H808
        ''' <summary>
        ''' PCAN-LAN interface, channel 9
        ''' </summary>
        Public Const PCANTP_LANBUS9 As TPCANTPHandle = &H809
        ''' <summary>
        ''' PCAN-LAN interface, channel 10
        ''' </summary>
        Public Const PCANTP_LANBUS10 As TPCANTPHandle = &H80A
        ''' <summary>
        ''' PCAN-LAN interface, channel 11
        ''' </summary>
        Public Const PCANTP_LANBUS11 As TPCANTPHandle = &H80B
        ''' <summary>
        ''' PCAN-LAN interface, channel 12
        ''' </summary>
        Public Const PCANTP_LANBUS12 As TPCANTPHandle = &H80C
        ''' <summary>
        ''' PCAN-LAN interface, channel 13
        ''' </summary>
        Public Const PCANTP_LANBUS13 As TPCANTPHandle = &H80D
        ''' <summary>
        ''' PCAN-LAN interface, channel 14
        ''' </summary>
        Public Const PCANTP_LANBUS14 As TPCANTPHandle = &H80E
        ''' <summary>
        ''' PCAN-LAN interface, channel 15
        ''' </summary>
        Public Const PCANTP_LANBUS15 As TPCANTPHandle = &H80F
        ''' <summary>
        ''' PCAN-LAN interface, channel 16
        ''' </summary>
        Public Const PCANTP_LANBUS16 As TPCANTPHandle = &H810
#End Region

#Region "FD Bit rate parameters"
        ''' <summary>
        ''' Clock frequency in Herz (80000000, 60000000, 40000000, 30000000, 24000000, 20000000)
        ''' </summary>
        Public Const PCANTP_BR_CLOCK As String = "f_clock"
        ''' <summary>
        ''' Clock frequency in Megaherz (80, 60, 40, 30, 24, 20)
        ''' </summary>
        Public Const PCANTP_BR_CLOCK_MHZ As String = "f_clock_mhz"
        ''' <summary>
        ''' Clock prescaler for nominal time quantum
        ''' </summary>
        Public Const PCANTP_BR_NOM_BRP As String = "nom_brp"
        ''' <summary>
        ''' TSEG1 segment for nominal bit rate in time quanta
        ''' </summary>
        Public Const PCANTP_BR_NOM_TSEG1 As String = "nom_tseg1"
        ''' <summary>
        ''' TSEG2 segment for nominal bit rate in time quanta
        ''' </summary>
        Public Const PCANTP_BR_NOM_TSEG2 As String = "nom_tseg2"
        ''' <summary>
        ''' Synchronization Jump Width for nominal bit rate in time quanta
        ''' </summary>
        Public Const PCANTP_BR_NOM_SJW As String = "nom_sjw"
        ''' <summary>
        ''' Sample point for nominal bit rate
        ''' </summary>
        Public Const PCANTP_BR_NOM_SAMPLE As String = "nom_sam"
        ''' <summary>
        ''' Clock prescaler for highspeed data time quantum
        ''' </summary>
        Public Const PCANTP_BR_DATA_BRP As String = "data_brp"
        ''' <summary>
        ''' TSEG1 segment for fast data bit rate in time quanta
        ''' </summary>
        Public Const PCANTP_BR_DATA_TSEG1 As String = "data_tseg1"
        ''' <summary>
        ''' TSEG2 segment for fast data bit rate in time quanta
        ''' </summary>
        Public Const PCANTP_BR_DATA_TSEG2 As String = "data_tseg2"
        ''' <summary>
        ''' Synchronization Jump Width for highspeed data bit rate in time quanta
        ''' </summary>
        Public Const PCANTP_BR_DATA_SJW As String = "data_sjw"
        ''' <summary>
        ''' Secondary sample point delay for highspeed data bit rate in cyles
        ''' </summary>
        Public Const PCANTP_BR_DATA_SAMPLE As String = "data_ssp_offset"
#End Region

#Region "Parameter values definition"
        ''' <summary>
        ''' No debug messages
        ''' </summary>
        Public Const PCANTP_DEBUG_NONE As Byte = 0
        ''' <summary>
        ''' Puts CAN debug messages to stdout
        ''' </summary>
        Public Const PCANTP_DEBUG_CAN As Byte = 1
        ''' <summary>
        ''' The Channel is illegal or not available
        ''' </summary>
        Public Const PCANTP_CHANNEL_UNAVAILABLE As Byte = 0
        ''' <summary>
        ''' The Channel is available
        ''' </summary>
        Public Const PCANTP_CHANNEL_AVAILABLE As Byte = 1
        ''' <summary>
        ''' The Channel is valid, and is being used
        ''' </summary>
        Public Const PCANTP_CHANNEL_OCCUPIED As Byte = 2

        ''' <summary>
        ''' if set Flow Control frame shall not use the WT flow status value
        ''' </summary>
        Public Const PCANTP_WFT_MAX_UNLIMITED As Byte = &H0

        ''' <summary>
        ''' An integer describing the Wait Frame Transmissions parameter.
        ''' </summary>
        Public Const PCANTP_WFT_MAX_DEFAULT As Byte = &H10

        ''' <summary>
        ''' Hide messages with type PCANTP_MESSAGE_INDICATION from CANTP_Read function
        ''' </summary>
        Public Const PCANTP_MSG_PENDING_HIDE As Byte = &H0
        ''' <summary>
        ''' Show messages with type PCANTP_MESSAGE_INDICATION from CANTP_Read function
        ''' </summary>
        Public Const PCANTP_MSG_PENDING_SHOW As Byte = &H1

        ''' <summary>
        ''' Uses CAN frame data optimization
        ''' </summary>
        Public Const PCANTP_CAN_DATA_PADDING_NONE As Byte = &H0
        ''' <summary>
        ''' Uses CAN frame data padding (default, i.e. CAN DLC = 8)
        ''' </summary>
        Public Const PCANTP_CAN_DATA_PADDING_ON As Byte = &H1

        ''' <summary>
        ''' Default value used if CAN data padding is enabled
        ''' </summary>
        Public Const PCANTP_CAN_DATA_PADDING_VALUE As Byte = &H55

        ''' <summary>
        ''' Disable reception of unformatted (NON-ISO-TP) CAN frames (default):
        ''' only ISO 15765 messages will be received
        ''' </summary>
        Public Const PCANTP_CAN_UNSEGMENTED_OFF = &H0
        ''' <summary>
        ''' Enable reception of unformatted (NON-ISO-TP) CAN frames:
        ''' received messages will be treated as either ISO 15765 Or as an unformatted CAN frame
        ''' </summary>
        Public Const PCANTP_CAN_UNSEGMENTED_ON = &H1
        ''' <summary>
        ''' Enable reception of unformatted (NON-ISO-TP) CAN frames:
        ''' received messages will be treated as ISO 15765, unformatted CAN frame, Or both (user will able to read fragmented CAN frames
        ''' </summary>
        Public Const PCANTP_CAN_UNSEGMENTED_ALL_FRAMES = &H2

        ''' <summary>
        ''' Default priority for ISO-TP messages (only available for normal fixed, mixed And enhanced addressing)
        ''' </summary>
        Public Const PCANTP_J1939_PRIORITY_DEFAULT = &H6

        ''' <summary>
        ''' Mask to retrieve the CAN ID type
        ''' </summary>
        Public Const PCANTP_ID_CAN_MASK = &HF
        ''' <summary>
        ''' Mask to check if the priority field is set
        ''' </summary>
        Public Const PCANTP_ID_CAN_IS_PRIORITY_MASK = &H10
        ''' <summary>
        ''' Mask to retrieve the J1939 priority
        ''' </summary>
        Public Const PCANTP_ID_CAN_PRIORITY_MASK = &HE0

        ''' <summary>
        ''' Constant defining that no CAN ID mapping exists
        ''' </summary>
        Public Const CAN_ID_NO_MAPPING As UInt32 = UInt32.MaxValue '&HFFFFFFFF

        ''' <summary>
        ''' max data length prior ISO 15765-2:2016 update
        ''' </summary>
        Public Const PCANTP_MESSAGE_MAX_LENGTH As UInt32 = &HFFF
#End Region

#Region "PCAN ISO-TP API Implementation"
        ''' <summary>
        ''' Initializes a PCANTP-Client based on a CANTP Channel
        ''' </summary>
        ''' <remarks>Only one PCANTP-Client can be initialized per CAN-Channel</remarks>
        ''' <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        ''' <param name="Baudrate">The CAN Hardware speed</param>
        ''' <param name="HwType">NON PLUG 'n PLAY: The type of hardware and operation mode</param>
        ''' <param name="IOPort">NON PLUG 'n PLAY: The I/O address for the parallel port</param>
        ''' <param name="Interrupt">NON PLUG 'n PLAY: Interrupt number of the parallel port</param>
        ''' <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-ISO-TP.dll", EntryPoint:="CANTP_Initialize")>
        Public Shared Function Initialize(
            <MarshalAs(UnmanagedType.U2)>
            ByVal CanChannel As TPCANTPHandle,
            <MarshalAs(UnmanagedType.U2)>
            ByVal Baudrate As TPCANTPBaudrate,
            <MarshalAs(UnmanagedType.U1)>
            ByVal HwType As TPCANTPHWType,
            ByVal IOPort As UInt32,
            ByVal Interrupt As UInt16) As TPCANTPStatus
        End Function


        ''' <summary>
        ''' Initializes a PCANTP-Client based on a CANTP Channel
        ''' </summary>
        ''' <remarks>Only one PCANTP-Client can be initialized per CAN-Channel</remarks>
        ''' <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        ''' <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        Public Shared Function Initialize(
            ByVal CanChannel As TPCANTPHandle,
            ByVal Baudrate As TPCANTPBaudrate)
            Return Initialize(CanChannel, Baudrate, 0, 0, 0)
        End Function


        ''' <summary>
        ''' Initializes a PCANTP-Client based on a CANTP Channel with FD support
        ''' </summary>
        ''' <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        ''' <param name="BitrateFD">The speed for the communication (FD bit rate string)</param>
        ''' <remarks>Only one PCANTP-Client can be initialized per CAN-Channel
        '''  See PCANTP_BR_* values
        ''' Bit rate string must follow the following construction rules:
        ''' * parameters And values must be separated by '='
        ''' * Couples of Parameter/value must be separated by ','
        ''' * Following Parameter must be filled out: f_clock, data_brp, data_sjw, data_tseg1, data_tseg2,
        '''   nom_brp, nom_sjw, nom_tseg1, nom_tseg2.
        ''' * Following Parameters are optional (Not used yet): data_ssp_offset, nom_samp</remarks>
        ''' <example>f_clock_mhz=80, nom_brp=1, nom_tseg1=63, nom_tseg2=16, nom_sjw=7, data_brp=4, data_tseg1=12, data_tseg2=7, data_sjw=1</example>
        ''' <returns>A TPCANTPStatus code. PCANTP_ERROR_OK Is returned on success</returns>
        <DllImport("PCAN-ISO-TP.dll", EntryPoint:="CANTP_InitializeFD")>
        Public Shared Function InitializeFD(
            <MarshalAs(UnmanagedType.U2)>
            ByVal CanChannel As TPCANTPHandle,
            ByVal BitrateFD As TPCANTPBitrateFD) As TPCANTPStatus
        End Function

        ''' <summary>
        ''' Uninitializes a PCANTP-Client initialized before
        ''' </summary>
        ''' <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        ''' <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-ISO-TP.dll", EntryPoint:="CANTP_Uninitialize")>
        Public Shared Function Uninitialize(
            <MarshalAs(UnmanagedType.U2)>
            ByVal CanChannel As TPCANTPHandle) As TPCANTPStatus
        End Function


        ''' <summary>
        ''' Resets the receive and transmit queues of a PCANTP-Client
        ''' </summary>
        ''' <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        ''' <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-ISO-TP.dll", EntryPoint:="CANTP_Reset")>
        Public Shared Function Reset(
            <MarshalAs(UnmanagedType.U2)>
            ByVal CanChannel As TPCANTPHandle) As TPCANTPStatus
        End Function

        ''' <summary>
        ''' Gets information about the internal BUS status of a PCANTP-Channel.
        ''' </summary>
        ''' <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        ''' <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-ISO-TP.dll", EntryPoint:="CANTP_GetStatus")>
        Public Shared Function GetStatus(
            <MarshalAs(UnmanagedType.U2)>
            ByVal CanChannel As TPCANTPHandle) As TPCANTPStatus
        End Function

        ''' <summary>
        ''' Reads a PCANTP message from the receive queue of a PCANTP-Client
        ''' </summary>
        ''' <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        ''' <param name="MessageBuffer">A TPCANTPMsg structure buffer to store the PUDS message</param>
        ''' <param name="TimestampBuffer">"A TPCANTPTimestamp structure buffer to get
        ''' the reception time of the message. If this value is not desired, this parameter
        ''' should be passed as NULL"</param>
        ''' <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-ISO-TP.dll", EntryPoint:="CANTP_Read")>
        Public Shared Function Read(
            <MarshalAs(UnmanagedType.U2)>
            ByVal CanChannel As TPCANTPHandle,
            ByRef MessageBuffer As TPCANTPMsg,
            ByRef TimestampBuffer As TPCANTPTimestamp) As TPCANTPStatus
        End Function

        <DllImport("PCAN-ISO-TP.dll", EntryPoint:="CANTP_Read")>
        Private Shared Function Read(
            <MarshalAs(UnmanagedType.U2)>
            ByVal CanChannel As TPCANTPHandle,
            ByRef MessageBuffer As TPCANTPMsg,
            ByVal TimestampBuffer As IntPtr) As TPCANTPStatus
        End Function
        ''' <summary>
        ''' Reads a PCANTP message from the receive queue of a PCANTP-Client
        ''' </summary>
        ''' <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        ''' <param name="MessageBuffer">A TPCANTPMsg structure buffer to store the PUDS message</param>
        ''' <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        Public Shared Function Read(
            <MarshalAs(UnmanagedType.U2)>
            ByVal CanChannel As TPCANTPHandle,
            ByRef MessageBuffer As TPCANTPMsg) As TPCANTPStatus
            Return Read(CanChannel, MessageBuffer, IntPtr.Zero)
        End Function

        ''' <summary>
        ''' Transmits a PCANTP message
        ''' </summary>
        ''' <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        ''' <param name="MessageBuffer">A TPCANTPMsg buffer with the message to be sent</param>
        ''' <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-ISO-TP.dll", EntryPoint:="CANTP_Write")>
        Public Shared Function Write(
            <MarshalAs(UnmanagedType.U2)>
            ByVal CanChannel As TPCANTPHandle,
            ByRef MessageBuffer As TPCANTPMsg) As TPCANTPStatus
        End Function

        ''' <summary>
        ''' Retrieves a PCANTP-Client value
        ''' </summary>
        ''' <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        ''' <param name="Parameter">The TPCANTPParameter parameter to get</param>
        ''' <param name="StringBuffer">Buffer for the parameter value</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-ISO-TP.dll", EntryPoint:="CANTP_GetValue")>
        Public Shared Function GetValue(
            <MarshalAs(UnmanagedType.U2)>
            ByVal CanChannel As TPCANTPHandle,
            <MarshalAs(UnmanagedType.U1)>
            ByVal Parameter As TPCANTPParameter,
            ByVal StringBuffer As StringBuilder,
            ByVal BufferLength As UInt32) As TPCANTPStatus
        End Function
        ''' <summary>
        ''' Retrieves a PCANTP-Client value
        ''' </summary>
        ''' <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        ''' <param name="Parameter">The TPCANTPParameter parameter to get</param>
        ''' <param name="NumericBuffer">Buffer for the parameter value</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-ISO-TP.dll", EntryPoint:="CANTP_GetValue")>
        Public Shared Function GetValue(
            <MarshalAs(UnmanagedType.U2)>
            ByVal CanChannel As TPCANTPHandle,
            <MarshalAs(UnmanagedType.U1)>
            ByVal Parameter As TPCANTPParameter,
            ByRef NumericBuffer As UInt32,
            ByVal BufferLength As UInt32) As TPCANTPStatus
        End Function
        ''' <summary>
        ''' Retrieves a PCANTP-Client value
        ''' </summary>
        ''' <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        ''' <param name="Parameter">The TPCANTPParameter parameter to get</param>
        ''' <param name="Buffer">Buffer for the parameter value</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-ISO-TP.dll", EntryPoint:="CANTP_GetValue")>
        Public Shared Function GetValue(
            <MarshalAs(UnmanagedType.U2)>
            ByVal CanChannel As TPCANTPHandle,
            <MarshalAs(UnmanagedType.U1)>
            ByVal Parameter As TPCANTPParameter,
            ByVal Buffer As Byte(),
            ByVal BufferLength As UInt32) As TPCANTPStatus
        End Function


        ''' <summary>
        ''' Configures or sets a PCANTP-Client value
        ''' </summary>
        ''' <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        ''' <param name="Parameter">The TPCANTPParameter parameter to set</param>
        ''' <param name="NumericBuffer">Buffer with the value to be set</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-ISO-TP.dll", EntryPoint:="CANTP_SetValue")>
        Public Shared Function SetValue(
            <MarshalAs(UnmanagedType.U2)>
            ByVal CanChannel As TPCANTPHandle,
            <MarshalAs(UnmanagedType.U1)>
            ByVal Parameter As TPCANTPParameter,
            ByRef NumericBuffer As UInt32,
            ByVal BufferLength As UInt32) As TPCANTPStatus
        End Function
        ''' <summary>
        ''' Configures or sets a PCANTP-Client value
        ''' </summary>
        ''' <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        ''' <param name="Parameter">The TPCANTPParameter parameter to set</param>
        ''' <param name="StringBuffer">Buffer with the value to be set</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-ISO-TP.dll", EntryPoint:="CANTP_SetValue")>
        Public Shared Function SetValue(
            <MarshalAs(UnmanagedType.U2)>
            ByVal CanChannel As TPCANTPHandle,
            <MarshalAs(UnmanagedType.U1)>
            ByVal Parameter As TPCANTPParameter,
            <MarshalAs(UnmanagedType.LPStr, SizeParamIndex:=3)>
            ByVal StringBuffer As String,
            ByVal BufferLength As UInt32) As TPCANTPStatus
        End Function
        ''' <summary>
        ''' Configures or sets a PCANTP-Client value
        ''' </summary>
        ''' <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        ''' <param name="Parameter">The TPCANTPParameter parameter to set</param>
        ''' <param name="Buffer">Buffer with the value to be set</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-ISO-TP.dll", EntryPoint:="CANTP_SetValue")>
        Public Shared Function SetValue(
            <MarshalAs(UnmanagedType.U2)>
            ByVal CanChannel As TPCANTPHandle,
            <MarshalAs(UnmanagedType.U1)>
            ByVal Parameter As TPCANTPParameter,
            ByVal Buffer As Byte(),
            ByVal BufferLength As UInt32) As TPCANTPStatus
        End Function

        ''' <summary>
        ''' Adds a user-defined PCAN-TP mapping between CAN ID and Network Address Information
        ''' </summary>
        ''' <remark>
        ''' Defining a mapping enables ISO-TP communication with 11BITS CAN ID or
        ''' with opened Addressing Formats (like PCANTP_FORMAT_NORMAL or PCANTP_FORMAT_EXTENDED).
        ''' </remark>
        ''' <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        ''' <param name="canID">The CAN ID to map</param>
        ''' <param name="canIDResponse">The CAN ID mapped response</param>
        ''' <param name="canIdType">CAN ID Type (11 or 29 bits, see PCANTP_ID_CAN_XXX)</param>
        ''' <param name="formatType">Addressing format (see PCANTP_FORMAT_XXX)</param>
        ''' <param name="msgType">Message type (remote or diagnostic, see PCANTP_MESSAGE_XXX)</param>
        ''' <param name="sourceAddr">Source Address</param>
        ''' <param name="targetAddr">Target Address</param>
        ''' <param name="targetType">Target Addressing Type (physical or functional, see PCANTP_ADDRESSING_XXX)</param>
        ''' <param name="remoteAddr">Address extension or Remote address</param>
        ''' <returns>A TPCANTPStatus code : PCANTP_ERROR_OK is returned on success,
        ''' PCANTP_ERROR_WRONG_PARAM states invalid Network Address Information parameters.</returns>
        <DllImport("PCAN-ISO-TP.dll", EntryPoint:="CANTP_AddMapping")>
        Public Shared Function AddMapping(
            <MarshalAs(UnmanagedType.U2)>
            ByVal CanChannel As TPCANTPHandle,
            ByVal canID As UInt32,
            ByVal canIDResponse As UInt32,
            <MarshalAs(UnmanagedType.U1)>
            ByVal canIdType As TPCANTPIdType,
            <MarshalAs(UnmanagedType.U1)>
            ByVal formatType As TPCANTPFormatType,
            <MarshalAs(UnmanagedType.U1)>
            ByVal msgType As TPCANTPMessageType,
            ByVal sourceAddr As Byte,
            ByVal targetAddr As Byte,
            <MarshalAs(UnmanagedType.U1)>
            ByVal targetType As TPCANTPAddressingType,
            ByVal remoteAddr As Byte) As TPCANTPStatus
        End Function

        ''' <summary>
        ''' Removes a user-defined PCAN-TP mapping between CAN ID and Network Address Information
        ''' </summary>
        ''' <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        ''' <param name="canID">The mapped CAN ID to remove</param>
        ''' <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-ISO-TP.dll", EntryPoint:="CANTP_RemoveMapping")>
        Public Shared Function RemoveMapping(
            <MarshalAs(UnmanagedType.U2)>
            ByVal CanChannel As TPCANTPHandle,
            ByVal canID As UInt32) As TPCANTPStatus
        End Function

        ''' <summary>
        ''' Returns a descriptive text of a given TPCANTPStatus error
        ''' code, in any desired language
        ''' </summary>
        ''' <remarks>The current languages available for translation are:
        ''' Neutral (0x00), German (0x07), English (0x09), Spanish (0x0A),
        ''' Italian (0x10) and French (0x0C)</remarks>
        ''' <param name="anError">A TPCANTPStatus error code</param>
        ''' <param name="Language">Indicates a 'Primary language ID'</param>
        ''' <param name="StringBuffer">Buffer for the text (must be at least 256 in length)</param>
        ''' <returns>A TPCANTPStatus error code</returns>
        <DllImport("PCAN-ISO-TP.dll", EntryPoint:="CANTP_GetErrorText")>
        Public Shared Function GetErrorText(
            <MarshalAs(UnmanagedType.U4)>
            ByVal anError As TPCANTPStatus,
            ByVal Language As UInt16,
            ByVal StringBuffer As StringBuilder) As TPCANTPStatus
        End Function

#End Region
    End Class
#End Region
End Namespace
